<?php
/*--------------------------------------------------------------
   Service.php 2021-08-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\DashboardWidget;

use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Data\PreferredCategoryStorage;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Data\PreferredPeriodStorage;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\Collections\Categories;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\Collections\SummarizedDataItems;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\Collections\Values;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\Entities\SummarizableTimespan;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\Entities\Value;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\Exceptions\InvalidPreferredCategoryException;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\Exceptions\InvalidPreferredPeriodException;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\Category;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\DateString;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\Name;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\PreferredCategory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\PreferredPeriod;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Model\ValueObjects\Result;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Services\Factories\SummarizableTimespanFactory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Services\Mapper\CompatibilityMapper;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial as SerialData;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial\Category as SerialCategory;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial\Item as SerialItem;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial\Item\Value as SerialItemValue;
use Gambio\Admin\Modules\Statistics\Services\StatisticsDashboardWidgetService;
use Gambio\Admin\Modules\Statistics\Services\StatisticsDataService;

class Service implements StatisticsDashboardWidgetService
{
    /**
     * @var SummarizableTimespanFactory
     */
    private $factory;
    
    /**
     * @var PreferredPeriodStorage
     */
    private $preferredPeriodStorage;
    
    /**
     * @var PreferredCategoryStorage
     */
    private $preferredCategoryStorage;
    
    /**
     * @var StatisticsDataService
     */
    private $service;
    
    /**
     * @var CompatibilityMapper
     */
    private $compatibilityMapper;
    
    
    /**
     * Constructor.
     */
    public function __construct(
        SummarizableTimespanFactory $factory,
        PreferredPeriodStorage $periodStorage,
        PreferredCategoryStorage $categoryStorage,
        StatisticsDataService $service,
        CompatibilityMapper $compatibilityMapper
    ) {
        $this->factory                  = $factory;
        $this->preferredPeriodStorage   = $periodStorage;
        $this->preferredCategoryStorage = $categoryStorage;
        $this->service                  = $service;
        $this->compatibilityMapper      = $compatibilityMapper;
    }
    
    
    /**
     * @inheritDoc
     */
    public function getStatistics(): Result
    {
        $timespan = $this->factory->createForTheWholePeriod();
        
        $categories = [];
        $results    = [];
        
        $serialData = [
            $this->service->getVisitorsByTimespan($timespan->startDate(), $timespan->endDate()),
            $this->service->getOrdersByTimespan($timespan->startDate(), $timespan->endDate()),
            $this->service->getSalesByTimespan($timespan->startDate(), $timespan->endDate()),
            $this->service->getConversionRatesByTimespan($timespan->startDate(), $timespan->endDate())
        ];
        
        /**
         * @var SerialData $data
         * @var SerialItem $item
         */
        foreach ($serialData as $data) {
            $values = new Values();
            $name   = '';
    
            $dates = [];
            foreach ($data->categories() as $key => $date) {
                $dates[$key] = $date->value();
            }
            
            foreach ($data->series() as $item) {
                $name = new Name($item->title()->value());
                
                /** @var SerialItemValue $value */
                foreach ($item->values() as $key => $value) {
                    $values[] = new Value(new DateString($dates[$key]), $value->value());
                }
            }
            
            $result       = $this->compatibilityMapper->mapToDataResultFactory($name->value(), $values);
            $categories[] = new Category($result->name(),
                                         $result->title(),
                                         $result->unit(),
                                         $result->style(),
                                         $result->minimumFractionDigits());
            
            $results[] = $result;
        }
        
        $statistics            = [];
        $timespanPresentations = $this->factory->createAll();
        
        /** @var SummarizableTimespan $timespan */
        foreach ($timespanPresentations as $timespan) {
            $statistics[] = $timespan->summarize(...$results);
        }
        
        return new Result(new SummarizedDataItems($statistics), new Categories($categories));
    }
    
    
    /**
     * @inheritDoc
     */
    public function updatePreferredPeriod(string $period): void
    {
        try {
            $preferredPeriod = new PreferredPeriod($period);
            $this->preferredPeriodStorage->setPreferredPeriod($preferredPeriod);
        } catch (InvalidPreferredPeriodException $exception) {
            unset($exception);
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function getPreferredPeriod(): ?PreferredPeriod
    {
        return $this->preferredPeriodStorage->getPreferredPeriod();
    }
    
    
    /**
     * @inheritDoc
     */
    public function updatePreferredCategory(string $category): void
    {
        try {
            $preferredCategory = new PreferredCategory($category);
            $this->preferredCategoryStorage->setPreferredCategory($preferredCategory);
        } catch (InvalidPreferredCategoryException $exception) {
            unset($exception);
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function getPreferredCategory(): ?PreferredCategory
    {
        return $this->preferredCategoryStorage->getPreferredCategory();
    }
}